// 3-Clause BSD license
/*-
*Copyright (C) 2019-2020 Intel Corporation
*SPDX-License-Identifier: BSD-3-Clause
 */
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <unistd.h>
#include <ifc_qdma_utils.h>
#include <ifc_reglib_osdep.h>
#include <ifc_libmqdma.h>

/* Block size */
#define MAXBUFLEN 64

/* Enable to load pattern */
#define LOAD_DATA
/* Disable to load data from input file */
#define DEFUALT_DATA
/* Enable to dump data */
#undef DEBUG

#if IFC_CONFIG_QDMA_NUM_DESC_PAGES_PER_QUEUE > 1
#define QDEPTH ((IFC_CONFIG_QDMA_NUM_DESC_PAGES_PER_QUEUE/2) * IFC_MCDMA_CONFIG_NUM_DESC_PER_PAGE)
#else
#define QDEPTH ((IFC_MCDMA_CONFIG_NUM_DESC_PER_PAGE/2) + 1)
#endif

enum perfq_status {
        PERFQ_CORE_ASSGN_FAILURE                                = -1,
        PERFQ_MALLOC_FAILURE                                    = -2,
        PERFQ_CMPLTN_NTFN_FAILURE                               = -3,
        PERFQ_DATA_VALDN_FAILURE                                = -4,
        PERFQ_FILE_VALDN_FAILURE                                = -5,
        PERFQ_DATA_PAYLD_FAILURE                                = -6,
        PERFQ_TASK_FAILURE                                      = -7,
        PERFQ_CH_INIT_FAILURE                                   = -8,
        PERFQ_SUCCESS                                           = 0,
        PERFQ_SOF                                               = 1,
        PERFQ_EOF                                               = 2,
        PERFQ_BOTH                                              = 3
};

#ifdef DEBUG
/*
 * In case if any anomalies observed at application, application
 * need to use this function, to dump memory in hex format
 */
static void qdma_hex_dump(unsigned char *base, int len)
{
        int i;

        for (i = 0; i < len; i++) {
                if ((i % 16) == 0)
                        fprintf(stderr, "\n%8lx ", (uint64_t) base + i);
                fprintf(stderr, "%02x ", base[i]);
        }
        fprintf (stderr, "\n");
}
#endif

#ifdef LOAD_DATA
#ifdef DEFUALT_DATA
#pragma GCC push_options
#pragma GCC optimize ("O0")
static uint32_t load_data(void *buf, size_t size, uint32_t pattern)
{
        unsigned int i;

        for (i = 0; i < (size / sizeof(uint32_t)); i++)
                *(((uint32_t *)buf) + i) = pattern++;

        return pattern;
}
#pragma GCC pop_options
#endif

#ifndef DEFUALT_DATA
static int load_data_frm_file(FILE *fp, void *buf)
{
	/* Read data from file */
	fread(buf, sizeof(void), MAXBUFLEN, fp);
	if (ferror(fp) != 0 ) {
		fputs("Error reading file", stderr);
		return -1;
	}
	return 0;
}
#endif
#endif

#ifdef DEFUALT_DATA
static int process_default_data(struct ifc_qdma_channel *qchnl,
				struct ifc_qdma_request **txd,
				uint32_t qdma_qdepth)
{
	struct ifc_qdma_request *d = NULL;
	void *tx_raw_pkt[32];
	uint32_t i = 0;
	uint32_t nr = 0;
	int ret = 0;

	/* pre-filling */
	for (i = 0; i < qdma_qdepth ; i++) {
		txd[i] = ifc_request_malloc(MAXBUFLEN);
		if (txd[i] == NULL)
			return -1;
		txd[i]->len = MAXBUFLEN;
#ifdef LOAD_DATA
#ifdef DEFUALT_DATA
		load_data(txd[i]->buf, MAXBUFLEN, 0x00);
#endif
#endif
		/* TODO: copy src data, for tx */
		ret = ifc_qdma_request_start(qchnl, IFC_QDMA_DIRECTION_TX, txd[i]);
		if (ret < 0)
			goto out;
#ifdef DEBUG
		qdma_hex_dump(txd[i]->buf, MAXBUFLEN);
#endif
	}
	printf("Pre-filling done for %d desc\n", qdma_qdepth);

	/* keep pushing same request */
	while (1) {
		nr = ifc_qdma_completion_poll(qchnl,
					      IFC_QDMA_DIRECTION_TX,
					      tx_raw_pkt,
					      ARRAY_SIZE(tx_raw_pkt));
		for (i = 0; i < nr; i++) {
			d = (struct ifc_qdma_request *)tx_raw_pkt[i];
#ifdef LOAD_DATA
			load_data(d->buf, MAXBUFLEN, 0x00);
#endif
			ret = ifc_qdma_request_start(qchnl, IFC_QDMA_DIRECTION_TX, d);
			if (ret < 0)
				goto out;
#ifdef DEBUG
			qdma_hex_dump(txd[i]->buf, MAXBUFLEN);
#endif
		}
		if (nr)
			printf("TEST: submitted %u TX descriptors\n", nr);
	}
out:
	for (i = 0; i < qdma_qdepth; i++)
		ifc_request_free(txd[i]);
	return 0;
}
#else
static int process_file_data(struct ifc_qdma_channel *qchnl,
			     struct ifc_qdma_request **txd, uint32_t qdma_qdepth)
{
	struct ifc_qdma_request *d = NULL;
	FILE *fp = NULL;
	void *tx_raw_pkt[32];
	uint32_t i = 0;
	uint32_t nr = 0;
	int ret = 0;
	const char *file_name = "load.bin";

	/* Read from file */
	fp = fopen(file_name, "rb");
	if (fp == NULL) {
		printf("ERR: Cannot open binary file with name: %s\n",file_name);
		return -1;
	}

	/* pre-filling */
	for (i = 0; i < qdma_qdepth ; i++) {
#ifdef LOAD_DATA
		/* check if data is available */
		if (feof(fp))
			break;
#endif
		txd[i] = ifc_request_malloc(MAXBUFLEN);
		if (txd[i] == NULL)
			return -1;
		txd[i]->len = MAXBUFLEN;
#ifdef LOAD_DATA
		ret = load_data_frm_file(fp, txd[i]->buf);
		if (ret < 0) {
			fclose(fp);
			goto out;
		}
#endif
		/* TODO: copy src data, for tx */
		ret = ifc_qdma_request_start(qchnl, IFC_QDMA_DIRECTION_TX, txd[i]);
		if (ret < 0)
			goto out;
#ifdef DEBUG
		qdma_hex_dump(txd[i]->buf, MAXBUFLEN);
#endif
	}
	printf("Pre-filling done for %d desc\n", qdma_qdepth);

	/* keep pushing same request till eof not reached */
	do {
		nr = ifc_qdma_completion_poll(qchnl,
					      IFC_QDMA_DIRECTION_TX,
					      tx_raw_pkt,
					      ARRAY_SIZE(tx_raw_pkt));
		for (i = 0; i < nr; i++) {
			d = (struct ifc_qdma_request *)tx_raw_pkt[i];
#ifdef LOAD_DATA
			ret = load_data_frm_file(fp, txd[i]->buf);
			if (ret < 0) {
				fclose(fp);
				goto out;
			}
#endif
			ret = ifc_qdma_request_start(qchnl, IFC_QDMA_DIRECTION_TX, d);
			if (ret < 0)
				goto out;
#ifdef DEBUG
			qdma_hex_dump(txd[i]->buf, MAXBUFLEN);
#endif
		}
		if (nr)
			printf("TEST: submitted %u TX descriptors\n", nr);
	} while (!feof(fp));
	printf("File ended\n");
	fclose(fp);
out:
	for (i = 0; i < qdma_qdepth; i++)
		ifc_request_free(txd[i]);

	return 0;
}
#endif

int main(void)
{
	struct ifc_qdma_request **txd;
	struct ifc_qdma_device *qdev;
	struct ifc_qdma_channel *qchnl;
	/* initialize with correct BDF value */
	const char *bdf = "0000:01:00.0";
	uint32_t block_size = MAXBUFLEN;
	uint32_t qdma_qdepth = QDEPTH;
	int ch = IFC_QDMA_AVAIL_CHNL_ARG;
	int ret;
	int port;

	ifc_app_start(bdf, block_size);

	port = ifc_mcdma_port_by_name(bdf); 
        if (port < 0) {
		goto out;
	}

	/* get DMA device */
	ret = ifc_qdma_device_get(port, &qdev, 128, IFC_CONFIG_QDMA_COMPL_PROC);
	if (ret)
		goto out;

	/* get number of channel */
	ret = ifc_num_channels_get(qdev);
	if (ret <= 0) {
		printf("no channels found in the dma device\n");
		goto out1;
	}

	/*
 	 * allocate a channel.
 	 * In case, if we have multi-threaded application, start calling this
 	 * functionality from channel specific thread
 	 */
	ret = ifc_qdma_channel_get(qdev, &qchnl, ch, IFC_QDMA_DIRECTION_BOTH);
	if (ret < 0)
		goto out1;

	/* pre-fill TX the descriptor queue */
	txd = malloc(qdma_qdepth * sizeof(*txd));
	if (txd == NULL)
		goto out1;

#ifdef DEFUALT_DATA
	ret = process_default_data(qchnl, txd, qdma_qdepth);
	if (ret < 0)
		goto out2;
#else
	ret = process_file_data(qchnl, txd, qdma_qdepth);
	if (ret < 0)
		goto out2;
#endif

out2:
	/* release your allocated channels */
	ifc_qdma_channel_put(qchnl, IFC_QDMA_DIRECTION_BOTH);
	free(txd);
out1:
	ifc_qdma_device_put(qdev);
out:
	ifc_app_stop();
	return 0;
}
